<?php

namespace App\Controllers\Backend;
use App\Controllers\BaseController;
use App\Models\Backend\SettingModel;
use App\Models\Backend\SettingEmailModel;
use App\Models\Backend\CaptchaProtectionModel;


class SettingController extends BaseController
{
    public function __construct()
    {
        $this->settingModel = new SettingModel();
        $this->captchaProtectionModel = new CaptchaProtectionModel();
        $this->settingEmailModel = new SettingEmailModel();
        $this->session = session();
    }


    //===========================================================//
	public function general_settings()
	{
	    $locale = $this->session->lang;
        $data['pageTitle'] = lang('Dashboard.generalSettings');
        $data['pageSubTitle'] = lang('Dashboard.generalSettings');
        $data['settings'] = $this->settingModel->select('settings_tbl.*')
            ->where('settings_tbl.locale', $locale)
            ->first();
        $data['captchaProtection'] = $this->captchaProtectionModel->select('captcha_protection_tbl.*')
            ->where('captcha_protection_tbl.id', 1)
            ->first();

        return view('\App\Views\backend\settings\general_settings_view', $data);
	}


    //===========================================================//
    public function update_configuration()
    {
        $active_tab = $this->request->getVar('active_tab');
        $rules = [
            'app_name'	                => ['rules'  => 'required|min_length[2]|validatePC', 
                                                'errors' => ['validateRTL' => lang('Validation.yourRTLInformationIsNotValid'),
                                                             'validateLicenseKey' => lang('Validation.yourLicenseKeyInformationIsNotValid'),
                                                            ],
                                           ],
            'app_description'	        => 'required|min_length[2]',
            'website_url'	            => 'required|valid_url',
            'installation_url'	        => 'required|valid_url',
            'terms_url'	                => 'required|valid_url',
            'android_app_dl_url'	    => 'required|valid_url',
            'android_app_dl_url2'	    => 'permit_empty|valid_url',
            'email_address'	            => 'required|valid_email',
            'phone1'	                => 'permit_empty|alpha_numeric_punct',
            'phone2'	                => 'permit_empty|alpha_numeric_punct',
            'phone3'	                => 'permit_empty|alpha_numeric_punct',
            'phone4'	                => 'permit_empty|alpha_numeric_punct',
            'working_hours'	            => 'permit_empty',
            'address'	                => 'permit_empty',
            'latitude'	                => 'permit_empty|alpha_numeric_punct',
            'longitude'	                => 'permit_empty|alpha_numeric_punct',
            'timezone'	                => 'required|timezone',
            'vat_rate'	                => 'required|integer',
            'vat_payable'	            => 'required|integer',
            'bank_account'	            => 'permit_empty',
            'licenses_and_certificates'	=> 'permit_empty',
            'calendar_type'	            => 'required',
            'switch_language'           => 'permit_empty|alpha_numeric_punct',
            'email_verification'        => 'permit_empty|alpha_numeric_punct',
            'registration'              => 'permit_empty|alpha_numeric_punct',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }  

        $locale = $this->session->lang;
        $show_advertising = $switch_language = $email_verification = $registration = 0;
        if($this->request->getVar('show_advertising') == "on") $show_advertising = 1;
        if($this->request->getVar('switch_language') == "on") $switch_language = 1;
        if($this->request->getVar('email_verification') == "on") $email_verification = 1;
        if($this->request->getVar('registration') == "on") $registration = 1;
        $data = [
            'app_name' 	        => $this->request->getVar('app_name'),
            'app_description' 	=> $this->request->getVar('app_description'),
            'website_url'       => $this->request->getVar('website_url'),
            'installation_url'  => $this->request->getVar('installation_url'),
            'terms_url'         => $this->request->getVar('terms_url'),
            'android_app_dl_url'  => $this->request->getVar('android_app_dl_url'),
            'android_app_dl_url2' => $this->request->getVar('android_app_dl_url2'),
            'email_address' 	=> $this->request->getVar('email_address'),
            'phone1' 	        => $this->request->getVar('phone1'),
            'phone2' 	        => $this->request->getVar('phone2'),
            'phone3' 	        => $this->request->getVar('phone3'),
            'phone4' 	        => $this->request->getVar('phone4'),
            'working_hours' 	=> $this->request->getVar('working_hours'),
            'address' 	        => $this->request->getVar('address'),
            'latitude' 	        => $this->request->getVar('latitude'),
            'longitude' 	    => $this->request->getVar('longitude'),
            'calendar_type' 	=> $this->request->getVar('calendar_type'),
            'timezone' 	        => $this->request->getVar('timezone'),
            'vat_rate' 	        => $this->request->getVar('vat_rate'),
            'vat_payable' 	    => $this->request->getVar('vat_payable'),
            'bank_account' 	    => $this->request->getVar('bank_account'),
            'licenses_and_certificates' => $this->request->getVar('licenses_and_certificates'),
            'show_advertising'  => $show_advertising,
            'switch_language'   => $switch_language,
            'email_verification'=> $email_verification,
            'registration' 	    => $registration,
        ];
        if (! $this->settingModel->update($locale, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
        }

        return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function update_social_media()
    {
        $active_tab = $this->request->getVar('active_tab');
        $rules = [
            'facebook'	            => 'permit_empty',
            'instagram'	            => 'permit_empty',
            'whatsapp'	            => 'permit_empty',
            'telegram'	            => 'permit_empty',
            'skype'	                => 'permit_empty',
            'twitter'	            => 'permit_empty',
            'linkedin'	            => 'permit_empty',
            'custom'	            => 'permit_empty',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $locale = $this->session->lang;
        $data = [
            'facebook' 	        => $this->request->getVar('facebook'),
            'instagram' 	    => $this->request->getVar('instagram'),
            'whatsapp' 	        => $this->request->getVar('whatsapp'),
            'telegram' 	        => $this->request->getVar('telegram'),
            'skype' 	        => $this->request->getVar('skype'),
            'twitter' 	        => $this->request->getVar('twitter'),
            'linkedin' 	        => $this->request->getVar('linkedin'),
            'custom' 	        => $this->request->getVar('custom'),
        ];

        if (! $this->settingModel->update($locale, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
        }

        return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function update_third_party_apps()
    {
        $active_tab = $this->request->getVar('active_tab');
        $rules = [
            'onesignal_app_id'          => 'permit_empty',
            'onesignal_rest_api_key'    => 'permit_empty',
            'youtube_api_key'           => 'permit_empty',
            'google_map_api_key'        => 'permit_empty',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }
        
        $db = db_connect();
        $locales = $db->query("SELECT locale FROM languages_tbl")->getResultArray();
        $db->close();

        foreach ($locales as $locale) {
            $data = [
                'onesignal_app_id' 	        => $this->request->getVar('onesignal_app_id'),
                'onesignal_rest_api_key' 	=> $this->request->getVar('onesignal_rest_api_key'),
                'youtube_api_key' 	        => $this->request->getVar('youtube_api_key'),
                'google_map_api_key' 	    => $this->request->getVar('google_map_api_key'),
            ];
    
            if (! $this->settingModel->update($locale, $data)) {
                return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
            }
        }

        return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function update_captcha_protection()
    {
        $active_tab = $this->request->getVar('active_tab');
        $rules = [
            'captcha_protection_website'    => 'permit_empty',
            'captcha_type'                  => 'required',
            'recaptcha_v2_site_key'         => 'permit_empty',
            'recaptcha_v2_secret_key'       => 'permit_empty',
            'recaptcha_v3_site_key'         => 'permit_empty',
            'recaptcha_v3_secret_key'       => 'permit_empty',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $recaptcha_v2_secret_key = $recaptcha_v3_secret_key = "";
        if(!empty($this->request->getVar('recaptcha_v2_secret_key')) || !empty($this->request->getVar('recaptcha_v3_secret_key'))) {
            $recaptcha_v2_secret_key = base64_encode(strrev(base64_encode($this->request->getVar('recaptcha_v2_secret_key'))));
            $recaptcha_v3_secret_key = base64_encode(strrev(base64_encode($this->request->getVar('recaptcha_v3_secret_key'))));
        }

        $captcha_protection_admin = $captcha_protection_website = $captcha_protection_pwa = 0;
        if($this->request->getVar('captcha_protection_admin') == "on") $captcha_protection_admin = 1;
        if($this->request->getVar('captcha_protection_website') == "on") $captcha_protection_website = 1;
        if($this->request->getVar('captcha_protection_pwa') == "on") $captcha_protection_pwa = 1;
        
        $data = [
            'captcha_protection_website' 	=> $captcha_protection_website,
            'captcha_type' 	                => $this->request->getVar('captcha_type'),
            'recaptcha_v2_site_key' 	    => $this->request->getVar('recaptcha_v2_site_key'),
            'recaptcha_v2_secret_key' 	    => $recaptcha_v2_secret_key,
            'recaptcha_v3_site_key' 	    => $this->request->getVar('recaptcha_v3_site_key'),
            'recaptcha_v3_secret_key' 	    => $recaptcha_v3_secret_key,
        ];

        $id = 1;
        if (!$this->captchaProtectionModel->update($id, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->captchaProtectionModel->errors());
        }

        return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function update_maintenance_mode()
    {
        $active_tab = $this->request->getPost('active_tab');
        $rules = [
            'maintenance_text'	    => 'required|min_length[3]',
            'maintenance_force_url'	=> 'permit_empty|valid_url',
            'maintenance_website'	=> 'permit_empty|alpha_numeric_punct',
            'maintenance_mobile'	=> 'permit_empty|alpha_numeric_punct',
            'maintenance_android'	=> 'permit_empty|alpha_numeric_punct',
            'maintenance_ios'	    => 'permit_empty|alpha_numeric_punct',
            'maintenance_other'	    => 'permit_empty|alpha_numeric_punct',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $locale = $this->session->lang;
        $maintenance_website = $maintenance_mobile = $maintenance_android = $maintenance_ios = $maintenance_other = 0;
        if($this->request->getVar('maintenance_website') == "on") $maintenance_website = 1;
        if($this->request->getVar('maintenance_mobile') == "on") $maintenance_mobile = 1;
        if($this->request->getVar('maintenance_android') == "on") $maintenance_android = 1;
        if($this->request->getVar('maintenance_ios') == "on") $maintenance_ios = 1;
        if($this->request->getVar('maintenance_other') == "on") $maintenance_other = 1;
        $data = [
            'maintenance_text' 	    => $this->request->getPost('maintenance_text'),
            'maintenance_force_url' => $this->request->getPost('maintenance_force_url'),
            'maintenance_website'   => $maintenance_website,
            'maintenance_mobile'    => $maintenance_mobile,
            'maintenance_android'   => $maintenance_android,
            'maintenance_ios'       => $maintenance_ios,
            'maintenance_other'     => $maintenance_other,
        ];

        if (! $this->settingModel->update($locale, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
        }

        return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function update_seo_tools()
    {
        $active_tab = $this->request->getVar('active_tab');
        $rules = [
            'seo_meta_description'	=> 'required|min_length[2]|max_length[155]',
            'seo_meta_keywords'	    => 'required|min_length[2]|max_length[255]',
            'seo_meta_author'	    => 'required|min_length[2]|max_length[40]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $locale = $this->session->lang;
        $data = [
            'seo_meta_description'  => $this->request->getPost('seo_meta_description'),
            'seo_meta_keywords'     => $this->request->getPost('seo_meta_keywords'),
            'seo_meta_author'       => $this->request->getPost('seo_meta_author'),
        ];

        if (! $this->settingModel->update($locale, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
        }

        return redirect()->to(base_url('dashboard/general-settings?active_tab='.$active_tab))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function logo_images()
    {
        $locale = $this->session->lang;
        $data['pageTitle'] = lang('Dashboard.logoImages');
        $data['pageSubTitle'] = lang('Dashboard.logoImages');
        $data['settings'] = $this->settingModel->select('settings_tbl.dashboard_logo, settings_tbl.light_logo_website, settings_tbl.dark_logo_website, settings_tbl.light_logo_mobile, settings_tbl.dark_logo_mobile,
                                                           settings_tbl.breadcrumbs_image, settings_tbl.light_ic_launcher, settings_tbl.dark_ic_launcher, settings_tbl.favicon')
            ->where('settings_tbl.locale', $locale)
            ->first();

        return view('\App\Views\backend\settings\logo_images_view', $data);
    }


    //===========================================================//
    public function update_dashboard_logo()
    {
        $locale = $this->session->lang;

        // Help: https://codeigniter4.github.io/userguide/libraries/uploaded_files.html
        $input = $this->validate([
            'image' => [
                'uploaded[dashboard_logo]',
                'ext_in[dashboard_logo,png,jpg,jpeg]',
                'is_image[dashboard_logo]',
                'max_size[dashboard_logo,200]',
                'errors' => [
                    'uploaded' => lang('Dashboard.pleaseSelectANewImage'),
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 200 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ]);

        if (!$input) {
            return redirect()->to(base_url('dashboard/logo-images'))->with('errors', $this->validator->getErrors());

        } else {
            // Remove previous image from disk
            $old_image_value = $this->settingModel->select('settings_tbl.dashboard_logo')->where('settings_tbl.locale', $locale)->first();
            @unlink('./backend/assets/uploads/admins/images/'.$old_image_value->dashboard_logo);

            $img = $this->request->getFile('dashboard_logo');
            $imgExtension = $img->getExtension();
            $newName = "dashboard_logo_$locale.$imgExtension";
            $img->move('./backend/assets/uploads/admins/images/', $newName);

            // Help: https://codeigniter4.github.io/userguide/libraries/images.html
            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(128, 128, 'center')
                ->save('./backend/assets/uploads/admins/images/'.$newName, $imgQuality);

            // Update image from DB
            $this->settingModel->set('dashboard_logo', $newName);
            $this->settingModel->where('locale', $locale);
            $this->settingModel->update();

            return redirect()->to(base_url('dashboard/logo-images'))->with('success', lang('Dashboard.imageUploadedSuccessfully'));
        }
    }


    //===========================================================//
    public function update_light_logo_website()
    {
        $locale = $this->session->lang;

        // Help: https://codeigniter4.github.io/userguide/libraries/uploaded_files.html
        $input = $this->validate([
            'image' => [
                'uploaded[light_logo_website]',
                'ext_in[light_logo_website,png,jpg,jpeg]',
                'is_image[light_logo_website]',
                'max_size[light_logo_website,1024]',
                'errors' => [
                    'uploaded' => lang('Dashboard.pleaseSelectANewImage'),
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 1024 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ]);

        if (!$input) {
            return redirect()->to(base_url('dashboard/logo-images'))->with('errors', $this->validator->getErrors());

        } else {
            // Remove previous image from disk
            $old_image_value = $this->settingModel->select('settings_tbl.light_logo_website')->where('settings_tbl.locale', $locale)->first();
            @unlink('./backend/assets/uploads/admins/images/'.$old_image_value->light_logo_website);

            $img = $this->request->getFile('light_logo_website');
            $imgExtension = $img->getExtension();
            $newName = "light_logo_website_$locale.$imgExtension";
            $img->move('./backend/assets/uploads/admins/images/', $newName);

            // Help: https://codeigniter4.github.io/userguide/libraries/images.html
            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(600, 180, 'center')
                ->save('./backend/assets/uploads/admins/images/'.$newName, $imgQuality);

            // Update image from DB
            $this->settingModel->set('light_logo_website', $newName);
            $this->settingModel->where('locale', $locale);
            $this->settingModel->update();

            return redirect()->to(base_url('dashboard/logo-images'))->with('success', lang('Dashboard.imageUploadedSuccessfully'));
        }
    }


    //===========================================================//
    public function update_dark_logo_website()
    {
        $locale = $this->session->lang;

        // Help: https://codeigniter4.github.io/userguide/libraries/uploaded_files.html
        $input = $this->validate([
            'image' => [
                'uploaded[dark_logo_website]',
                'ext_in[dark_logo_website,png,jpg,jpeg]',
                'is_image[dark_logo_website]',
                'max_size[dark_logo_website,1024]',
                'errors' => [
                    'uploaded' => lang('Dashboard.pleaseSelectANewImage'),
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 1024 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ]);

        if (!$input) {
            return redirect()->to(base_url('dashboard/logo-images'))->with('errors', $this->validator->getErrors());

        } else {
            // Remove previous image from disk
            $old_image_value = $this->settingModel->select('settings_tbl.dark_logo_website')->where('settings_tbl.locale', $locale)->first();
            @unlink('./backend/assets/uploads/admins/images/'.$old_image_value->dark_logo_website);

            $img = $this->request->getFile('dark_logo_website');
            $imgExtension = $img->getExtension();
            $newName = "dark_logo_website_$locale.$imgExtension";
            $img->move('./backend/assets/uploads/admins/images/', $newName);

            // Help: https://codeigniter4.github.io/userguide/libraries/images.html
            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(600, 180, 'center')
                ->save('./backend/assets/uploads/admins/images/'.$newName, $imgQuality);

            // Update image from DB
            $this->settingModel->set('dark_logo_website', $newName);
            $this->settingModel->where('locale', $locale);
            $this->settingModel->update();

            return redirect()->to(base_url('dashboard/logo-images'))->with('success', lang('Dashboard.imageUploadedSuccessfully'));
        }
    }


    //===========================================================//
    public function update_light_logo_mobile()
    {
        $locale = $this->session->lang;

        // Help: https://codeigniter4.github.io/userguide/libraries/uploaded_files.html
        $input = $this->validate([
            'image' => [
                'uploaded[light_logo_mobile]',
                'ext_in[light_logo_mobile,png,jpg,jpeg]',
                'is_image[light_logo_mobile]',
                'max_size[light_logo_mobile,1024]',
                'errors' => [
                    'uploaded' => lang('Dashboard.pleaseSelectANewImage'),
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 1024 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ]);

        if (!$input) {
            return redirect()->to(base_url('dashboard/logo-images'))->with('errors', $this->validator->getErrors());

        } else {
            // Remove previous image from disk
            $old_image_value = $this->settingModel->select('settings_tbl.light_logo_mobile')->where('settings_tbl.locale', $locale)->first();
            @unlink('./backend/assets/uploads/admins/images/'.$old_image_value->light_logo_mobile);

            $img = $this->request->getFile('light_logo_mobile');
            $imgExtension = $img->getExtension();
            $newName = "light_logo_mobile_$locale.$imgExtension";
            $img->move('./backend/assets/uploads/admins/images/', $newName);

            // Help: https://codeigniter4.github.io/userguide/libraries/images.html
            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(600, 180, 'center')
                ->save('./backend/assets/uploads/admins/images/'.$newName, $imgQuality);

            // Update image from DB
            $this->settingModel->set('light_logo_mobile', $newName);
            $this->settingModel->where('locale', $locale);
            $this->settingModel->update();

            return redirect()->to(base_url('dashboard/logo-images'))->with('success', lang('Dashboard.imageUploadedSuccessfully'));
        }
    }


    //===========================================================//
    public function update_dark_logo_mobile()
    {
        $locale = $this->session->lang;

        // Help: https://codeigniter4.github.io/userguide/libraries/uploaded_files.html
        $input = $this->validate([
            'image' => [
                'uploaded[dark_logo_mobile]',
                'ext_in[dark_logo_mobile,png,jpg,jpeg]',
                'is_image[dark_logo_mobile]',
                'max_size[dark_logo_mobile,1024]',
                'errors' => [
                    'uploaded' => lang('Dashboard.pleaseSelectANewImage'),
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 1024 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ]);

        if (!$input) {
            return redirect()->to(base_url('dashboard/logo-images'))->with('errors', $this->validator->getErrors());

        } else {
            // Remove previous image from disk
            $old_image_value = $this->settingModel->select('settings_tbl.dark_logo_mobile')->where('settings_tbl.locale', $locale)->first();
            @unlink('./backend/assets/uploads/admins/images/'.$old_image_value->dark_logo_mobile);

            $img = $this->request->getFile('dark_logo_mobile');
            $imgExtension = $img->getExtension();
            $newName = "dark_logo_mobile_$locale.$imgExtension";
            $img->move('./backend/assets/uploads/admins/images/', $newName);

            // Help: https://codeigniter4.github.io/userguide/libraries/images.html
            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(600, 180, 'center')
                ->save('./backend/assets/uploads/admins/images/'.$newName, $imgQuality);

            // Update image from DB
            $this->settingModel->set('dark_logo_mobile', $newName);
            $this->settingModel->where('locale', $locale);
            $this->settingModel->update();

            return redirect()->to(base_url('dashboard/logo-images'))->with('success', lang('Dashboard.imageUploadedSuccessfully'));
        }
    }


    //===========================================================//
    public function update_breadcrumbs_image()
    {
        $locale = $this->session->lang;

        // Help: https://codeigniter4.github.io/userguide/libraries/uploaded_files.html
        $input = $this->validate([
            'image' => [
                'uploaded[breadcrumbs_image]',
                'ext_in[breadcrumbs_image,png,jpg,jpeg]',
                'is_image[breadcrumbs_image]',
                'max_size[breadcrumbs_image,2048]',
                'errors' => [
                    'uploaded' => lang('Dashboard.pleaseSelectANewImage'),
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 2048 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ]);

        if (!$input) {
            return redirect()->to(base_url('dashboard/logo-images'))->with('errors', $this->validator->getErrors());

        } else {
            // Remove previous image from disk
            $old_image_value = $this->settingModel->select('settings_tbl.breadcrumbs_image')->where('settings_tbl.locale', $locale)->first();
            @unlink('./backend/assets/uploads/admins/images/'.$old_image_value->breadcrumbs_image);

            $img = $this->request->getFile('breadcrumbs_image');
            $imgExtension = $img->getExtension();
            $newName = "breadcrumbs_image_$locale.$imgExtension";
            $img->move('./backend/assets/uploads/admins/images/', $newName);

            // Help: https://codeigniter4.github.io/userguide/libraries/images.html
            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(1920, 440, 'center')
                ->save('./backend/assets/uploads/admins/images/'.$newName, $imgQuality);

            // Update image from DB
            $this->settingModel->set('breadcrumbs_image', $newName);
            $this->settingModel->where('locale', $locale);
            $this->settingModel->update();

            return redirect()->to(base_url('dashboard/logo-images'))->with('success', lang('Dashboard.imageUploadedSuccessfully'));
        }
    }


    //===========================================================//
    public function update_light_ic_launcher()
    {
        $locale = $this->session->lang;

        // Help: https://codeigniter4.github.io/userguide/libraries/uploaded_files.html
        $input = $this->validate([
            'image' => [
                'uploaded[light_ic_launcher]',
                'ext_in[light_ic_launcher,png,jpg,jpeg]',
                'is_image[light_ic_launcher]',
                'max_size[light_ic_launcher,1024]',
                'errors' => [
                    'uploaded' => lang('Dashboard.pleaseSelectANewImage'),
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 1024 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ]);

        if (!$input) {
            return redirect()->to(base_url('dashboard/logo-images'))->with('errors', $this->validator->getErrors());

        } else {
            // Remove previous image from disk
            $old_image_value = $this->settingModel->select('settings_tbl.light_ic_launcher')->where('settings_tbl.locale', $locale)->first();
            @unlink('./backend/assets/uploads/admins/images/'.$old_image_value->light_ic_launcher);

            $img = $this->request->getFile('light_ic_launcher');
            $imgExtension = $img->getExtension();
            $newName = "light_ic_launcher_$locale.$imgExtension";
            $img->move('./backend/assets/uploads/admins/images/', $newName);

            // Help: https://codeigniter4.github.io/userguide/libraries/images.html
            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(512, 512, 'center')
                ->save('./backend/assets/uploads/admins/images/'.$newName, $imgQuality);

            // Update image from DB
            $this->settingModel->set('light_ic_launcher', $newName);
            $this->settingModel->where('locale', $locale);
            $this->settingModel->update();

            return redirect()->to(base_url('dashboard/logo-images'))->with('success', lang('Dashboard.imageUploadedSuccessfully'));
        }
    }


    //===========================================================//
    public function update_dark_ic_launcher()
    {
        $locale = $this->session->lang;

        // Help: https://codeigniter4.github.io/userguide/libraries/uploaded_files.html
        $input = $this->validate([
            'image' => [
                'uploaded[dark_ic_launcher]',
                'ext_in[dark_ic_launcher,png,jpg,jpeg]',
                'is_image[dark_ic_launcher]',
                'max_size[dark_ic_launcher,1024]',
                'errors' => [
                    'uploaded' => lang('Dashboard.pleaseSelectANewImage'),
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 1024 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ]);

        if (!$input) {
            return redirect()->to(base_url('dashboard/logo-images'))->with('errors', $this->validator->getErrors());

        } else {
            // Remove previous image from disk
            $old_image_value = $this->settingModel->select('settings_tbl.dark_ic_launcher')->where('settings_tbl.locale', $locale)->first();
            @unlink('./backend/assets/uploads/admins/images/'.$old_image_value->light_ic_launcher);

            $img = $this->request->getFile('dark_ic_launcher');
            $imgExtension = $img->getExtension();
            $newName = "dark_ic_launcher_$locale.$imgExtension";
            $img->move('./backend/assets/uploads/admins/images/', $newName);

            // Help: https://codeigniter4.github.io/userguide/libraries/images.html
            $imgPath = $img->getTempName();
            $imgQuality = 90;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(512, 512, 'center')
                ->save('./backend/assets/uploads/admins/images/'.$newName, $imgQuality);

            // Update image from DB
            $this->settingModel->set('dark_ic_launcher', $newName);
            $this->settingModel->where('locale', $locale);
            $this->settingModel->update();

            return redirect()->to(base_url('dashboard/logo-images'))->with('success', lang('Dashboard.imageUploadedSuccessfully'));
        }
    }


    //===========================================================//
    public function update_favicon()
    {
        $locale = $this->session->lang;

        // Help: https://codeigniter4.github.io/userguide/libraries/uploaded_files.html
        $input = $this->validate([
            'image' => [
                'uploaded[favicon]',
                'ext_in[favicon,png,jpg,jpeg,ico]',
                'is_image[favicon]',
                'max_size[favicon,100]',
                'errors' => [
                    'uploaded' => lang('Dashboard.pleaseSelectANewImage'),
                    'max_size' => lang('Dashboard.theMaximumAcceptableSizeIs', [ 100 ]),
                    'ext_in' => lang('Dashboard.onlyImagesWithPngJpgJpegExtensionAreSupported')
                ]
            ],
        ]);

        if (!$input) {
            return redirect()->to(base_url('dashboard/logo-images'))->with('errors', $this->validator->getErrors());

        } else {
            // Remove previous image from disk
            $old_image_value = $this->settingModel->select('settings_tbl.favicon')->where('settings_tbl.locale', $locale)->first();
            @unlink('./backend/assets/uploads/admins/images/'.$old_image_value->favicon);

            $img = $this->request->getFile('favicon');
            $imgExtension = $img->getExtension();
            $newName = "favicon.$imgExtension";
            $img->move('./backend/assets/uploads/admins/images/', $newName);

            // Help: https://codeigniter4.github.io/userguide/libraries/images.html
            $imgPath = $img->getTempName();
            $imgQuality = 100;
            $watermark = "";
            $image = \Config\Services::image()
                ->withFile($imgPath.$newName)
                ->text($watermark, [
                    'color'      => '#f9f9f9',
                    'opacity'    => 0.7,
                    'withShadow' => false,
                    'hAlign'     => 'center', /* hAlign Horizontal alignment: left, center, right */
                    'vAlign'     => 'middle', /* vAlign Vertical alignment: top, middle, bottom */
                    'fontSize'   => 25,
                    'fontPath'   => ''
                ])
                ->fit(128, 128, 'center')
                ->save($newName, $imgQuality);

            // Update image from DB
            $this->settingModel->set('favicon', $newName);
            $this->settingModel->where('locale', $locale);
            $this->settingModel->update();

            return redirect()->to(base_url('dashboard/logo-images'))->with('success', lang('Dashboard.imageUploadedSuccessfully'));
        }
    }


    //===========================================================//
    public function email_settings()
    {
        $data['pageTitle'] = lang('Dashboard.emailSettings');
        $data['pageSubTitle'] = lang('Dashboard.sendingEmailSettings');
        $data['emailSettings'] = $this->settingEmailModel->select('settings_email_tbl.*')
            ->where('settings_email_tbl.id', 1)
            ->first();

        return view('\App\Views\backend\settings\email_settings_view', $data);
    }


    //===========================================================//
    public function perform_email_settings()
    {
        $rules = [
            'from_email'	            => 'required|min_length[6]|max_length[60]|valid_email',
            'no_reply_email'	        => 'required|min_length[6]|max_length[60]|valid_email',
            'from_name'	                => 'required|min_length[2]',
            'smtp_host'	                => 'permit_empty',
            'smtp_user'	                => 'permit_empty',
            'smtp_pass'	                => 'permit_empty',
            'smtp_port'	                => 'permit_empty',
            'protocol'	                => 'required',
            'smtp_crypto'	            => 'required',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/email-settings'))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $data = [
            'id' 	            => 1,
            'from_email' 	    => $this->request->getVar('from_email'),
            'no_reply_email'    => $this->request->getVar('no_reply_email'),
            'from_name' 	    => $this->request->getVar('from_name'),
            'smtp_host' 	    => $this->request->getVar('smtp_host'),
            'smtp_user' 	    => $this->request->getVar('smtp_user'),
            'smtp_pass' 	    => base64_encode(strrev(base64_encode($this->request->getVar('smtp_pass')))),
            'smtp_port' 	    => $this->request->getVar('smtp_port'),
            'protocol' 	        => $this->request->getVar('protocol'),
            'smtp_crypto' 	    => $this->request->getVar('smtp_crypto'),
        ];
        if (! $this->settingEmailModel->save($data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingEmailModel->errors());
        }

        return redirect()->to(base_url('dashboard/email-settings'))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function api_docs()
    {
        $locale = $this->session->lang;
        $data['pageTitle'] = lang('Dashboard.apiDocs');
        $data['pageSubTitle'] = lang('Dashboard.apiDocs');

        return view('\App\Views\backend\settings\api_docs_view', $data);
    }


    //===========================================================//
    public function website_templates()
    {
        $directory = APPPATH.'/Views/frontend';
        $data['scanDirectory'] = scandir($directory);
        $data['pageTitle'] = lang('Dashboard.websiteTemplates');
        $data['pageSubTitle'] = lang('Dashboard.websiteTemplates');
        return view('\App\Views\backend\settings\website_templates_view', $data);
    }


    //===========================================================//
    public function select_template($template_name = NULL)
    {
        // Update template for all locale
        $this->settingModel->query("UPDATE settings_tbl SET template_name = '$template_name'");

        // Update template for current locale
        /*$locale = $this->session->lang;
        $template_name = service('uri')->getSegment(3);
        $data = [
            'template_name' 	    => $template_name,
        ];
        if (! $this->settingModel->update($locale, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
        }*/

        return redirect()->to(base_url('dashboard/website-templates'))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function static_page_terms()
    {
        $locale = $this->session->lang;
        $data['pageTitle'] = lang('Dashboard.termsAndConditions');
        $data['pageSubTitle'] = lang('Dashboard.termsAndConditionsPage');
        $data['settings'] = $this->settingModel->select('settings_tbl.page_terms')
            ->where('settings_tbl.locale', $locale)
            ->first();

        return view('\App\Views\backend\pages\static_page_terms_view', $data);
    }

    //===========================================================//
    public function perform_edit_static_page_terms()
    {
        $rules = [
            'page_terms'        => 'required|min_length[5]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/static-page-terms'))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $locale = $this->session->lang;
        $data = [
            'page_terms' 	    => $this->request->getVar('page_terms'),
        ];
        if (! $this->settingModel->update($locale, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
        }

        return redirect()->to(base_url('dashboard/static-page-terms'))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function static_page_privacy()
    {
        $locale = $this->session->lang;
        $data['pageTitle'] = lang('Dashboard.privacyPolicy');
        $data['pageSubTitle'] = lang('Dashboard.privacyPolicyPage');
        $data['settings'] = $this->settingModel->select('settings_tbl.page_privacy')
            ->where('settings_tbl.locale', $locale)
            ->first();

        return view('\App\Views\backend\pages\static_page_privacy_view', $data);
    }


    //===========================================================//
    public function perform_edit_static_page_privacy()
    {
        $rules = [
            'page_privacy'        => 'required|min_length[5]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/static-page-privacy'))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $locale = $this->session->lang;
        $data = [
            'page_privacy' 	    => $this->request->getVar('page_privacy'),
        ];
        if (! $this->settingModel->update($locale, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
        }

        return redirect()->to(base_url('dashboard/static-page-privacy'))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function static_page_about()
    {
        $locale = $this->session->lang;
        $data['pageTitle'] = lang('Dashboard.aboutUs');
        $data['pageSubTitle'] = lang('Dashboard.aboutUsPage');
        $data['settings'] = $this->settingModel->select('settings_tbl.page_about')
            ->where('settings_tbl.locale', $locale)
            ->first();

        return view('\App\Views\backend\pages\static_page_about_view', $data);
    }


    //===========================================================//
    public function perform_edit_static_page_about()
    {
        $rules = [
            'page_about'        => 'required|min_length[5]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/static-page-about'))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $locale = $this->session->lang;
        $data = [
            'page_about' 	    => $this->request->getVar('page_about'),
        ];
        if (! $this->settingModel->update($locale, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
        }

        return redirect()->to(base_url('dashboard/static-page-about'))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function static_page_advertisement()
    {
        $locale = $this->session->lang;
        $data['pageTitle'] = lang('Dashboard.advertisement');
        $data['pageSubTitle'] = lang('Dashboard.advertisementPage');
        $data['settings'] = $this->settingModel->select('settings_tbl.page_advertisement')
            ->where('settings_tbl.locale', $locale)
            ->first();

        return view('\App\Views\backend\pages\static_page_advertisement_view', $data);
    }


    //===========================================================//
    public function perform_edit_static_page_advertisement()
    {
        $rules = [
            'page_advertisement'        => 'required|min_length[5]',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url('dashboard/static-page-advertisement'))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $locale = $this->session->lang;
        $data = [
            'page_advertisement' 	    => $this->request->getVar('page_advertisement'),
        ];
        if (! $this->settingModel->update($locale, $data)) {
            return redirect()->back()->withInput()->with('errors', $this->settingModel->errors());
        }

        return redirect()->to(base_url('dashboard/static-page-advertisement'))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }


    //===========================================================//
    public function snippets_list()
    {
        $locale = $this->session->lang;

        $db = db_connect();
        // Fetch snippets code
        $builder = $db->table('custom_codes_tbl');
        $builder->select('custom_codes_tbl.*, custom_codes_place_tbl.title AS place_title');
        $builder->join('custom_codes_place_tbl', 'custom_codes_place_tbl.id = custom_codes_tbl.place_id');
        $builder->orderBy('custom_codes_tbl.id', 'ASC');
        $data['snippets'] = $builder->get()->getResult();
        $db->close();

        $data['pageTitle'] = lang('Dashboard.codeSnippets');
        $data['pageSubTitle'] = lang('Dashboard.codeSnippets');
        $data['settings'] = $this->settingModel->select('settings_tbl.*')
            ->where('settings_tbl.locale', $locale)
            ->first();

        return view('\App\Views\backend\settings\snippets_list_view', $data);
    }


    //===========================================================//
    public function edit_snippet()
    {
        $id = service('uri')->getSegment(3);

        $db = db_connect();
        // Fetch snippets code
        $builder = $db->table('custom_codes_tbl');
        $builder->where('custom_codes_tbl.id', $id);
        $data['snippet'] = $builder->get()->getResult()[0];
        $db->close();

        if(is_null($data['snippet']))
            return redirect()->back()->with('error', lang('Dashboard.notFound'));

        $data['pageTitle'] = lang('Dashboard.editSnippet');
        $data['pageSubTitle'] = lang('Dashboard.editSnippet');
        return view('\App\Views\backend\settings\edit_snippet_view', $data);
    }


    //--------------------------------------------------------------------//
    public function perform_edit_snippet()
    {
        $id = $this->request->getVar('id');
        $active = 0;
        if($this->request->getPost('active') == "on")
            $active = 1;

        $rules = [
            'id'	        => 'integer|required',
            'place_id'	    => 'integer|required',
            'title'	        => 'required|min_length[2]',
            'codes'	        => 'permit_empty',
            'active'	    => 'permit_empty',
        ];

        if (!$this->validate($rules)) {
            return redirect()->to(base_url("dashboard/edit-snippet/$id"))
                ->withInput()
                ->with('errors', $this->validator->getErrors());
        }

        $data = [
            'id'  	        => $id,
            'place_id'  	=> $this->request->getVar('place_id'),
            'title'  	    => $this->request->getVar('title'),
            'codes' 	    => $this->request->getVar('codes'),
            'active'     	=> $active,
        ];

        $db = db_connect();
        $builder = $db->table('custom_codes_tbl');
        $builder->replace($data);
        $db->close();

        return redirect()->to(base_url('dashboard/snippets-list'))->with('success', lang('Dashboard.dataSuccessfullyUpdated'));
    }

}
