<?php

namespace App\Controllers\Auth;
use App\Controllers\BaseController;
use App\Models\Backend\ActivityModel;
use App\Models\Backend\SettingEmailModel;
use App\Models\Backend\UserModel;
use App\Models\Backend\CurrencyModel;
use App\Models\Backend\WalletModel;

class RegisterController extends BaseController
{
    public function __construct()
    {
        $this->userModel = new UserModel();
        $this->activityModel = new ActivityModel();
        $this->settingEmailModel = new SettingEmailModel();
        $this->currencyModel = new CurrencyModel();
        $this->walletModel = new WalletModel();
        $this->session = session();
    }


    //===========================================================//
	public function register()
	{
        $data['pageTitle'] = lang('Dashboard.register');
        return view('\App\Views\auth\register_view', $data);
	}


	//===========================================================//
    public function perform_register()
    {
        // Validate reCAPTCHA
        $recaptchaType = $this->request->getVar('g-recaptcha-type');
        $recaptchaToken = $this->request->getVar('g-recaptcha-response');
        if(!empty($recaptchaType) && !empty($recaptchaToken)) {
            helper('google_recaptcha');
            if (!validateRecaptcha($recaptchaToken, $recaptchaType)) {
                return $this->response->setJSON([
                    //'errors' => ['recaptcha' => 'reCAPTCHA validation failed. Please try again.'],
                    'messageType'       => 'toastr', // sweetalert OR toastr
                    'messageIcon'       => 'error', // success , error , warning , info
                    'messageTitle'      => lang('Dashboard.error'),
                    'messageBody'       => lang('Dashboard.reCAPTCHAValidationFailedPleaseTryAgain'),
                    ])->setStatusCode(400); // 400 Bad Request
            }
        }

        // Validation the form
        $rules = [
            'first_name' => 'required|min_length[2]|max_length[35]',
            'last_name' => 'required|min_length[2]|max_length[35]',
            'email' => 'required|min_length[6]|max_length[60]|valid_email|is_unique[users_tbl.email]',
            'password' => 'required|min_length[6]|max_length[120]',
            'password_confirm' => 'required|matches[password]',
            'terms' => 'required',
        ];

        $errors = [
            'first_name' => [
                'required' => lang('Validation.pleaseEnterYourFirstName'),
                'min_length' => lang('Validation.theFirstNameFieldMustBeAtLeastCharactersInLength', [2]),
                'max_length' => lang('Validation.theFirstNameFieldCannotExceedCharactersInLength', [35]),
            ],
            'last_name' => [
                'required' => lang('Validation.pleaseEnterYourLastName'),
                'min_length' => lang('Validation.theLastNameFieldMustBeAtLeastCharactersInLength', [2]),
                'max_length' => lang('Validation.theLastNameFieldCannotExceedCharactersInLength', [35]),
            ],
            'email' => [
                'required' => lang('Validation.pleaseEnterYourEmailAddress'),
                'min_length' => lang('Validation.theEmailFieldMustBeAtLeastCharactersInLength', [6]),
                'max_length' => lang('Validation.theEmailFieldCannotExceedCharactersInLength', [60]),
                'valid_email' => lang('Validation.yourEmailAddressIsNotValid'),
                'is_unique' => lang('Validation.theEmailFieldMustContainAUniqueValue'),
            ],
            'password' => [
                'required' => lang('Validation.pleaseEnterYourPassword'),
                'min_length' => lang('Validation.thePasswordFieldMustBeAtLeastCharactersInLength', [6]),
                'max_length' => lang('Validation.thePasswordFieldCannotExceedCharactersInLength', [120]),
            ],
            'password_confirm' => [
                'required' => lang('Validation.pleaseEnterYourConfirmPassword'),
                'matches' => lang('Validation.yourPasswordFieldsDoesNotMatched'),
            ],
            'terms' => [
                'required' => lang('Validation.theTermsOfServicesFieldIsRequired'),
            ],
        ];

        if (!$this->validate($rules, $errors)) {
            return $this->response->setJSON([
                'messageType'   => 'toastr', // sweetalert OR toastr
                'messageIcon'   => 'error', // success , error , warning , info
                'messageTitle'  => lang('Dashboard.error'),
                'errors'        => $this->validator->getErrors(),
                ])->setStatusCode(400); // 400 Bad Request
        }

        // Valid submission
        $activate_hash = random_string('alnum', 32);
        $newData = [
            'first_name'        => $this->request->getVar('first_name'),
            'last_name'         => $this->request->getVar('last_name'),
            'email'             => $this->request->getVar('email'),
            'password'          => $this->request->getVar('password'),
            'image'	            => 'profile-image.png',
            'device_type_id'	=> 2,
            'dark_mode'     	=> 0,
            'api_key_access'    => 0,
            'activate_hash' 	=> $activate_hash,
        ];
        $this->userModel->save($newData);

        $last_id = $this->userModel->insertID();

        // Create default wallet
        $currency = $this->currencyModel->where('default', 1)->first();
        if($currency) {
            $walletData = [
                'user_id'       => $last_id,
                'currency_id'   => $currency->id,
                'title'         => lang('Dashboard.defaultWallet'),
                'credit'        => 0,
                'status'        => 1,
            ];
            $this->walletModel->save($walletData);
        }

        // User activity
        $activity = [
            'user_id' => $last_id,
            'icon'  => 'fas fa-user-plus',
            'background'  => 'bg-success',
            'title'  => lang('Dashboard.register'),
            'body'  => lang('Dashboard.theUserBecameAMemberThroughTheWebsite'),
            'ip'  => $_SERVER['REMOTE_ADDR'],
            'agent'  => $_SERVER['HTTP_USER_AGENT'],
            'time'  => time(),
        ];
        $this->activityModel->save($activity);

        // Send welcome & activation emails
        $settingsEmail = $this->settingEmailModel->select('settings_email_tbl.*')->where('settings_email_tbl.id', 1)->first();
        $subject1 = lang('Dashboard.emailAddressActivation');
        $subject2 = lang('Dashboard.accountInformation');
        $full_name = $this->request->getVar('first_name')." ".$this->request->getVar('last_name');
        $password = lang('Dashboard.hidden');
        helper('email');
        @send_activation_email($settingsEmail, $this->request->getVar('email'), $subject1, $activate_hash);
        @send_account_information_email($settingsEmail, $subject2, $full_name, $this->request->getVar('email'), $password);

        $this->session->setFlashdata('success', lang('Dashboard.membershipCompletedSuccessfully'));
        return $this->response->setJSON([
            'redirectUrl' => base_url('auth/login'),
            'messageType' => 'toastr', // sweetalert OR toastr
            'messageIcon' => 'success', // success , error , warning , info
            'messageTitle' => lang('Dashboard.success'),
            'messageBody' => lang('Dashboard.membershipCompletedSuccessfully')
        ]);
    }


    //===========================================================//
    public function activate_account()
    {
        // Check token
        $user = $this->userModel->where('activate_hash', $this->request->getGet('token'))->where('status_id ', 2)->first();

        if (is_null($user)) {
            return redirect()->to(base_url('auth/login'))->with('error', lang('Dashboard.thereIsNoUserWithThisActivationCode'));
        }

        // Update user account to active
        $newData = [
            'id' => $user->id,
            'status_id' => 1,
        ];
        $this->userModel->save($newData);

        $activity = [
            'user_id' => $user->id,
            'icon'  => 'fas fa-envelope',
            'background'  => 'bg-gray',
            'title'  => lang('Dashboard.activation'),
            'body'  => lang('Dashboard.emailAddressVerifiedSuccessfully'),
            'ip'  => $_SERVER['REMOTE_ADDR'],
            'agent'  => $_SERVER['HTTP_USER_AGENT'],
            'time'  => time(),
        ];
        $this->activityModel->save($activity);

        return redirect()->to(base_url('auth/login'))->with('success', lang('Dashboard.emailAddressVerifiedSuccessfully'));
    }

}
